#!/usr/bin/env bash

# SPDX-FileCopyrightText: 2022 Patrick Spek <p.spek@tyil.nl>
#
# SPDX-License-Identifier: AGPL-3.0-or-later

# OS independent service management.
svc() {
	local system="bashtard/svc"

	local service
	local action

	action=$1 ; shift
	service="$(config "svc.$1")" ; shift

	if [[ -z $service ]]
	then
		crit "$system" "No service name for $service"
		return 1
	fi

	if [[ "$(type -t "svc_$action")" != "function" ]]
	then
		crit "$system" "Invalid service manager action $action"
		return 1
	fi

	"svc_$action" "$service"
}

svc_disable() {
	local system="bashtard/svc/disable"

	local service=$1

	case "${BASHTARD_PLATFORM[key]}" in
		linux-alpine_linux) set -- /sbin/rc-update del "$service" ;;
		linux-gentoo)       set -- /sbin/rc-update del "$service" ;;
		linux-*)            set -- systemctl disable "$service" ;;
		*)
			crit "$system" "No service manager configured for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}

svc_enable() {
	local system="bashtard/svc/enable"

	local service=$1
	local rc

	case "${BASHTARD_PLATFORM[key]}" in
		freebsd)
			rc="$(printf '%s_enable="YES"' "$service")"

			if [[ ! -f "/etc/rc.conf.d/$service" ]] || ! grep -Fqx "$rc" "/etc/rc.conf.d/$service"
			then
				printf "%s\n" "$rc" >> "/etc/rc.conf.d/$service"
			fi

			return 0

			;;
		linux-alpine_linux) set -- /sbin/rc-update add "$service" ;;
		linux-gentoo)       set -- /sbin/rc-update add "$service" ;;
		linux-*)            set -- systemctl enable "$service" ;;
		*)
			crit "$system" "No service manager configured for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}

svc_reload() {
	local system="bashtard/svc/reload"

	local service=$1

	case "${BASHTARD_PLATFORM[key]}" in
		freebsd)            set -- service "$service" reload ;;
		linux-alpine_linux) set -- /sbin/rc-service "$service" reload ;;
		linux-gentoo)       set -- /sbin/rc-service "$service" reload ;;
		linux-*)            set -- systemctl reload "$service" ;;
		*)
			crit "$system" "No service manager configured for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}

svc_restart() {
	local system="bashtard/svc/restart"

	local service=$1

	case "${BASHTARD_PLATFORM[key]}" in
		freebsd)            set -- service "$service" restart ;;
		linux-alpine_linux) set -- /sbin/rc-service "$service" restart ;;
		linux-gentoo)       set -- /sbin/rc-service "$service" restart ;;
		linux-*)            set -- systemctl restart "$service" ;;
		*)
			crit "$system" "No service manager configured for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}

svc_start() {
	local system="bashtard/svc/start"

	local service=$1

	case "${BASHTARD_PLATFORM[key]}" in
		freebsd)            set -- service "$service" start ;;
		linux-alpine_linux) set -- /sbin/rc-service "$service" start ;;
		linux-gentoo)       set -- /sbin/rc-service "$service" start ;;
		linux-*)            set -- systemctl start "$service" ;;
		*)
			crit "$system" "No service manager configured for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}

svc_stop() {
	local system="bashtard/svc/stop"

	local service=$1

	case "${BASHTARD_PLATFORM[key]}" in
		freebsd)            set -- service "$service" stop ;;
		linux-alpine_linux) set -- /sbin/rc-service "$service" stop ;;
		linux-gentoo)       set -- /sbin/rc-service "$service" stop ;;
		linux-*)            set -- systemctl stop "$service" ;;
		*)
			crit "$system" "No service manager configured for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}
