#!/usr/bin/env bash

# SPDX-FileCopyrightText: 2023 Patrick Spek <p.spek@tyil.nl>
#
# SPDX-License-Identifier: AGPL-3.0-or-later

subcommand()
{
	local results
	local widths

	declare -A results
	declare -A widths=(
		[load]=0
		[memory]=0
		[node]=0
		[storage]=0
		[uptime]=0
	)

	if [[ ! -d "$BASHTARD_ETCDIR/hosts.d" ]]
	then
		crit "$BASHTARD_NAME/ssh" "Could not find hosts file at $BASHTARD_ETCDIR/hosts.d"
		return 3
	fi

	chgdir "$BASHTARD_ETCDIR/hosts.d"

	for node in *
	do
		local user
		local host

		user="$(config_for "$node" "bashtard.ssh.user" "$USER")"
		host="$(config_for "$node" "bashtard.ssh.host")"

		if [[ -z "$host" ]]
		then
			crit "$BASHTARD_NAME/ssh" "bashtard.ssh.host is not configured for $node"
			continue
		fi

		if [[ "$node" == "${BASHTARD_PLATFORM[fqdn]}" ]]
		then
			results+=(["$node"]="$("$BASHTARD_BIN" sysinfo)")
		else
			debug "$BASHTARD_NAME/ssh" "$user@$node ($host) > bashtard sysinfo"
			# shellcheck disable=SC2029
			results+=(["$node"]="$(ssh "$user@$host" "bashtard sysinfo" 2>/dev/null)")
		fi

		unset user
		unset host
	done

	# Check widths
	for node in "${!results[@]}"
	do
		node_load="$(grep '^load' <<< "${results["$node"]}" | sed 's/[^ ]* *//')"
		node_memory="$(grep '^memory' <<< "${results["$node"]}" | sed 's/[^ ]* *//')"
		node_storage="$(grep '^storage' <<< "${results["$node"]}" | sed 's/[^ ]* *//')"
		node_uptime="$(grep '^uptime' <<< "${results["$node"]}" | sed 's/[^ ]* *//')"

		width_load="$(wc -c <<< "$node_load")"
		width_memory="$(wc -c <<< "$node_memory")"
		width_node="$(wc -c <<< "$node")"
		width_storage="$(wc -c <<< "$node_storage")"
		width_uptime="$(wc -c <<< "$node_uptime")"

		if (( widths[load] < width_load ))
		then
			widths[load]=$width_load
		fi

		if (( widths[memory] < width_memory ))
		then
			widths[memory]=$width_memory
		fi

		if (( widths[node] < width_node ))
		then
			widths[node]=$width_node
		fi

		if (( widths[storage] < width_storage ))
		then
			widths[storage]=$width_storage
		fi

		if (( widths[uptime] < width_uptime ))
		then
			widths[uptime]=$width_uptime
		fi
	done

	# Print results
	printf "$(tput bold)%-${widths[node]}s  %-${widths[load]}s %-${widths[memory]}s %-${widths[storage]}s %-${widths[uptime]}s$(tput sgr0)\n" \
		"Node" \
		"Load" \
		"Memory" \
		"Storage" \
		"Uptime"

	for node in "${!results[@]}"
	do
		printf "%-${widths[node]}s %${widths[load]}s %${widths[memory]}s %${widths[storage]}s %${widths[uptime]}s\n" \
			"$node" \
			"$(grep '^load' <<< "${results["$node"]}" | sed 's/[^ ]* *//')" \
			"$(grep '^memory' <<< "${results["$node"]}" | sed 's/[^ ]* *//')" \
			"$(grep '^storage' <<< "${results["$node"]}" | sed 's/[^ ]* *//')" \
			"$(grep '^uptime' <<< "${results["$node"]}" | sed 's/[^ ]* *//')"
	done | sort
}
