#!/usr/bin/env bash

# SPDX-FileCopyrightText: 2022 Patrick Spek <p.spek@tyil.nl>
#
# SPDX-License-Identifier: AGPL-3.0-or-later

# OS independent service management.
svc() {
	local system="bashtard/svc"

	local service
	local action

	action=$1 ; shift
	service="$(config "svc.$1")" ; shift

	if [[ -z $service ]]
	then
		crit "$system" "No service name for $service"
		return 1
	fi

	if [[ "$(type -t "svc_$action")" != "function" ]]
	then
		crit "$system" "Invalid service manager action $action"
		return 1
	fi

	"svc_$action" "$service"
}

svc_disable() {
	local system="bashtard/svc/disable"

	local service=$1

	case "${BASHTARD_PLATFORM[os]}" in
		linux)
			case "${BASHTARD_PLATFORM[init]}" in
				openrc)  set -- /sbin/rc-update del "$service" ;;
				systemd) set -- systemctl disable "$service" ;;
				*)
					crit "$system" "No service manager support to disable with ${BASHTARD_PLATFORM[init]}"
					return 1
			esac
			;;
		*)
			crit "$system" "No service manager support to enable for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}

svc_enable() {
	local system="bashtard/svc/enable"

	local service=$1
	local rc

	case "${BASHTARD_PLATFORM[os]}" in
		linux)
			case "${BASHTARD_PLATFORM[init]}" in
				openrc)  set -- /sbin/rc-update add "$service" ;;
				systemd) set -- systemctl enable "$service" ;;
				*)
					crit "$system" "No service manager support to enable with ${BASHTARD_PLATFORM[init]}"
					return 1
			esac
			;;
		freebsd)
			rc="$(printf '%s_enable="YES"' "$service")"

			if [[ ! -f "/etc/rc.conf.d/$service" ]] || ! grep -Fqx "$rc" "/etc/rc.conf.d/$service"
			then
				printf "%s\n" "$rc" >> "/etc/rc.conf.d/$service"
			fi

			return 0
			;;
		*)
			crit "$system" "No service manager support to enable for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}

svc_reload() {
	local system="bashtard/svc/reload"

	local service=$1

	case "${BASHTARD_PLATFORM[os]}" in
		linux)
			case "${BASHTARD_PLATFORM[init]}" in
				openrc)  set -- /sbin/rc-service "$service" reload ;;
				systemd) set -- systemctl reload "$service" ;;
				*)
					crit "$system" "No service manager support to reload with ${BASHTARD_PLATFORM[init]}"
					return 1
			esac
			;;
		freebsd)
			set -- service "$service" reload
			;;
		*)
			crit "$system" "No service manager support to reload for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}

svc_restart() {
	local system="bashtard/svc/restart"

	local service=$1

	case "${BASHTARD_PLATFORM[os]}" in
		linux)
			case "${BASHTARD_PLATFORM[init]}" in
				openrc)  set -- /sbin/rc-service "$service" restart ;;
				systemd) set -- systemctl restart "$service" ;;
				*)
					crit "$system" "No service manager support to restart with ${BASHTARD_PLATFORM[init]}"
					return 1
			esac
			;;
		freebsd)
			set -- service "$service" restart
			;;
		*)
			crit "$system" "No service manager support to restart for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}

svc_start() {
	local system="bashtard/svc/start"

	local service=$1

	case "${BASHTARD_PLATFORM[os]}" in
		linux)
			case "${BASHTARD_PLATFORM[init]}" in
				openrc)  set -- /sbin/rc-service "$service" start ;;
				systemd) set -- systemctl start "$service" ;;
				*)
					crit "$system" "No service manager support to start with ${BASHTARD_PLATFORM[init]}"
					return 1
			esac
			;;
		freebsd)
			set -- service "$service" start
			;;
		*)
			crit "$system" "No service manager support to start for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}

svc_stop() {
	local system="bashtard/svc/stop"

	local service=$1

	case "${BASHTARD_PLATFORM[os]}" in
		linux)
			case "${BASHTARD_PLATFORM[init]}" in
				openrc)  set -- /sbin/rc-service "$service" stop ;;
				systemd) set -- systemctl stop "$service" ;;
				*)
					crit "$system" "No service manager support to stop with ${BASHTARD_PLATFORM[init]}"
					return 1
			esac
			;;
		freebsd)
			set -- service "$service" stop
			;;
		*)
			crit "$system" "No service manager support to stop for ${BASHTARD_PLATFORM[key]}"
			return 1
	esac

	notice "$system" "$*"
	# shellcheck disable=SC2068
	$@
}
