#!/usr/bin/env bash

# SPDX-FileCopyrightText: 2022 Patrick Spek <p.spek@tyil.nl>
#
# SPDX-License-Identifier: AGPL-3.0-or-later

subcommand() {
	local load_1
	local load_5
	local load_15
	local memory_free
	local memory_total
	local memory_used
	local storage_free
	local storage_total
	local storage_used
	local uptime

	# Set the variables which are compatible on every POSIX-compliant system
	storage_used="$(df | awk '/[0-9]+/ { sum += $3 } END { print sum }')"
	storage_free="$(df | awk '/[0-9]+/ { sum += $4 } END { print sum }')"
	storage_total=$(( storage_used + storage_free ))

	# And do platform-specific magic
	case "${BASHTARD_PLATFORM[os]}" in
		freebsd)
			load_1="$(uptime | awk -F' *,? *' '{ print $(NF-2) }')"
			load_5="$(uptime | awk -F' *,? *' '{ print $(NF-1) }')"
			load_15="$(uptime | awk -F' *,? *' '{ print $NF }')"
			memory_total=$(( $(sysctl -n hw.physmem) / 1024 ))
			memory_used=$(( ( $(sysctl -n vm.stats.vm.v_active_count) * $(sysctl -n hw.pagesize) ) / 1024 ))
			memory_free=$(( memory_total - memory_used))
			uptime=$(( "$(date +%s)" - "$(sysctl -a | awk '/^kern.boottime/ { print substr($5, 0, length($5)-1) }')" ))
			;;
		*)
			load_1="$(awk '{ print $1 }' < /proc/loadavg)"
			load_5="$(awk '{ print $2 }' < /proc/loadavg)"
			load_15="$(awk '{ print $3 }' < /proc/loadavg)"
			memory_total="$(awk '/MemTotal/ { print $2 }' < /proc/meminfo)"
			memory_free="$(awk '/MemAvailable/ { print $2 }' < /proc/meminfo)"
			memory_used=$(( memory_total - memory_free ))
			uptime="$(awk -F. '{ print $1 }' < /proc/uptime)"
			;;
	esac

	# For any value that isn't set, just default to 0 to avoid all sorts of errors
	[[ -z "$load_1" ]] && load_1=0
	[[ -z "$load_5" ]] && load_5=0
	[[ -z "$load_15" ]] && load_15=0
	[[ -z "$memory_free" ]] && memory_free=0
	[[ -z "$memory_total" ]] && memory_total=0
	[[ -z "$memory_used" ]] && memory_used=0
	[[ -z "$storage_free" ]] && storage_free=0
	[[ -z "$storage_total" ]] && storage_total=0
	[[ -z "$storage_used" ]] && storage_used=0
	[[ -z "$uptime" ]] && uptime=0

	# Print the values that can be set by package maintainers
	printf "%-15s %s\n" "etcdir" "$BASHTARD_ETCDIR"
	printf "%-15s %s\n" "libdir" "$BASHTARD_LIBDIR"
	printf "%-15s %s\n" "sharedir" "$BASHTARD_SHAREDIR"

	# Print all the discovered platform information
	for key in "${!BASHTARD_PLATFORM[@]}"
	do
		printf "%-15s %s\n" "$key" "${BASHTARD_PLATFORM[$key]}"
	done

	# Print fun little extras
	printf "%-15s %0.1fGi / %0.1fGi\n" "memory" \
		"$(awk '{ print($1 / 1024 / 1024) }' <<< "$memory_used")" \
		"$(awk '{ print($1 / 1024 / 1024) }' <<< "$memory_total")"
	printf "%-15s %0.1fGb / %0.1fGb\n" "storage" \
		"$(awk '{ print($1 / 1024 / 1024) }' <<< "$storage_used")" \
		"$(awk '{ print($1 / 1024 / 1024) }' <<< "$storage_total")"
	printf "%-15s %0.2f %0.2f %0.2f\n" "load" \
		"$load_1" \
		"$load_5" \
		"$load_15"
	printf "%-15s %dd %02dh %02dm %02ds\n" "uptime" \
		"$(( uptime / 60 / 60 / 24 ))" \
		"$(( uptime / 60 / 60 % 24 ))" \
		"$(( uptime / 60 % 60 ))" \
		"$(( uptime % 60 ))"
}
