// Copyright (C) 2025  Patrick "tyil" Spek <p.spek@tyil.nl>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

package config

import (
	"fmt"
	"io/ioutil"
	"log/slog"
	"net/http"
	"os"
	"slices"
	"strings"

	"git.tyil.nl/go/env.git"
	"go.yaml.in/yaml/v3"
)

// Global functions

func Load() {
	config = defaults

	path := env.String("DASHLAB_CONFIG", "config.yaml")
	data, err := ioutil.ReadFile(path)
	if err != nil {
		slog.Error(
			"Could not read configuration file",
			"path", path,
			"error", err.Error(),
		)

		return
	}

	err = yaml.Unmarshal(data, &config)
	if err != nil {
		slog.Error(
			"Could not parse configuration file",
			"path", path,
			"error", err.Error(),
		)

		return
	}
}

func Get() Config {
	return config
}

// ConfigApp functions

func (this ConfigApp) GetIconName() string {
	if this.Icon.Name != "" {
		return this.Icon.Name
	}

	return strings.ToLower(this.Title)
}

func (this ConfigApp) GetIconSet() string {
	if this.Icon.Set != "" {
		return this.Icon.Set
	}

	return ""
}

func (this ConfigApp) GetIconUrl() string {
	if this.Icon.Url != "" {
		return this.Icon.Url
	}

	icon_name := this.GetIconName()
	icon_set := this.GetIconSet()
	icon_sets := []string{
		"simple-icons",
		"cbi",
	}

	// Try to find an icon in the sets available
	if icon_set == "" {
		for i := 0; i < len(icon_sets); i++ {
			_, err := os.Stat(fmt.Sprintf("./icon/%s/%s.svg", icon_sets[i], icon_name))
			if err == nil {
				icon_set = icon_sets[i]
				break
			}
		}
	}

	// Return an icon if found
	if icon_set != "" {
		return fmt.Sprintf(
			"/icon/%s/%s.svg",
			icon_set,
			icon_name,
		)
	}

	// Return the fallback icon
	return fmt.Sprintf(
		"/icon/%s/%s.svg",
		config.Defaults.IconSet,
		config.Defaults.IconName,
	)
}

// ConfigAuth functions

func (this ConfigAuth) Username(req *http.Request) string {
	return req.Header.Get(this.UserHeader)
}

func (this ConfigAuth) Groups(req *http.Request) []string {
	return strings.Split(req.Header.Get(this.GroupsHeader), this.Separator)
}

func (this ConfigAuth) GroupsSorted(req *http.Request) []string {
	groups := this.Groups(req)

	slices.Sort(groups)

	return groups
}

func (this ConfigAuth) Can(groups []string, req *http.Request) bool {
	allowed := this.Groups(req)

	// If no explicit groups are set, implicit allow is given
	if len(groups) < 1 {
		return true
	}

	for i := 0; i < len(groups); i++ {
		if slices.Contains(allowed, groups[i]) {
			return true
		}
	}

	return false
}

// ConfigSection functions

func (this ConfigSection) CountVisible(auth ConfigAuth, req *http.Request) int {
	count := 0

	for i := 0; i < len(this.Apps); i++ {
		if auth.Can(this.Apps[0].Groups, req) {
			count++
		}
	}

	return count
}

func (this ConfigSection) IsVisible(auth ConfigAuth, req *http.Request) bool {
	return 0 < this.CountVisible(auth, req)
}

// ConfigServer functions

func (this ConfigServer) Listen() string {
	return fmt.Sprintf("%s:%d", this.Address, this.Port)
}
