// Copyright (C) 2025  Patrick "tyil" Spek <p.spek@tyil.nl>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

package main

import (
	"log/slog"
	"net/http"
	"os"
	"time"

	"dashlab/config"
	"dashlab/handlers"
	"dashlab/middlewares"
)

func main() {
	middleware := MiddlewareChain{
		middlewares.Logging,
	}
	mux := http.NewServeMux()

	// Load configuration
	config.Load()

	// Set up logger
	logger := slog.New(slog.NewTextHandler(os.Stdout, &slog.HandlerOptions{
		Level: slog.LevelDebug,
	}))
	slog.SetDefault(logger)

	// Include ConfigLoader middleware if requested
	if config.Get().ReloadMiddleware {
		slog.Info("Appending ConfigLoader middleware")
		middleware = append(middleware, middlewares.ConfigLoader)
	}

	// Set routes
	mux.Handle("GET /icon/", http.StripPrefix("/icon", http.FileServer(http.Dir("./icon"))))
	mux.Handle("GET /style.css", middleware.thenFunc(handlers.Stylesheet))
	mux.Handle("GET /whoami", middleware.thenFunc(handlers.WhoAmI))
	mux.Handle("GET /about", middleware.thenFunc(handlers.About))
	mux.Handle("GET /-/healthz", middleware.thenFunc(handlers.Health))
	mux.Handle("GET /", middleware.thenFunc(handlers.Index))

	// Start HTTP server
	server := &http.Server{
		Handler:      mux,
		Addr:         config.Get().Server.Listen(),
		WriteTimeout: 15 * time.Second,
		ReadTimeout:  15 * time.Second,
	}

	slog.Info(
		"Starting DashLab",
		"address", config.Get().Server.Address,
		"port", config.Get().Server.Port,
	)

	server.ListenAndServe()
}
